<?php
require __DIR__ . '/../../bootstrap.php';

header('Content-Type: application/json; charset=utf-8');

// Helper for JSON response
function json_response($data, $status = 200) {
    http_response_code($status);
    echo json_encode($data);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    json_response(['error' => 'Method not allowed'], 405);
}

$input = json_decode(file_get_contents('php://input'), true);

$authEmail = $input['auth_email'] ?? '';
$authPassword = $input['auth_password'] ?? '';
$newEmail = isset($input['new_email']) ? trim($input['new_email']) : null;
$newPassword = isset($input['new_password']) ? $input['new_password'] : null;

if (empty($authEmail) || empty($authPassword)) {
    json_response(['error' => 'البريد وكلمة المرور مطلوبان'], 401);
}

try {
    // Authenticate
    $stmt = db()->prepare('SELECT id, password_hash, email FROM admins WHERE email = :email LIMIT 1');
    $stmt->execute(['email' => $authEmail]);
    $admin = $stmt->fetch();

    if (!$admin || !password_verify($authPassword, (string)$admin['password_hash'])) {
        json_response(['error' => 'بيانات الدخول غير صحيحة'], 401);
    }

    $adminId = $admin['id'];
    $updates = [];
    $params = ['id' => $adminId];

    // Check email update
    if ($newEmail !== null && $newEmail !== '' && $newEmail !== $admin['email']) {
        // Check uniqueness
        $check = db()->prepare('SELECT id FROM admins WHERE email = :email AND id <> :id');
        $check->execute(['email' => $newEmail, 'id' => $adminId]);
        if ($check->fetch()) {
            json_response(['error' => 'البريد الإلكتروني مستخدم بالفعل'], 400);
        }
        $updates[] = "email = :new_email";
        $params['new_email'] = $newEmail;
    }

    // Check password update
    if ($newPassword !== null && $newPassword !== '') {
        if (strlen($newPassword) < 6) {
            json_response(['error' => 'كلمة المرور الجديدة يجب أن تكون 6 أحرف على الأقل'], 400);
        }
        $updates[] = "password_hash = :new_hash";
        $params['new_hash'] = password_hash($newPassword, PASSWORD_DEFAULT);
    }

    if (empty($updates)) {
        json_response(['success' => true, 'message' => 'لم يتم تغيير أي بيانات']);
    }

    $sql = "UPDATE admins SET " . implode(', ', $updates) . " WHERE id = :id";
    $stmt = db()->prepare($sql);
    $stmt->execute($params);

    json_response([
        'success' => true,
        'message' => 'تم تحديث البيانات بنجاح',
        'email_updated' => isset($params['new_email']),
        'password_updated' => isset($params['new_hash'])
    ]);

} catch (Exception $e) {
    json_response(['error' => 'حدث خطأ في الخادم: ' . $e->getMessage()], 500);
}
